<?php
// admin/reports.php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../models/Record.php';
require_once '../models/Expense.php';
require_once '../models/Branch.php';

// Check if user is logged in and is admin
redirectIfNotLoggedIn();
redirectIfNotAdmin();

$database = new Database();
$db = $database->getConnection();
$record = new Record($db);
$expense = new Expense($db);
$branch = new Branch($db);

// Get all branches for filter
$result = $branch->getAll();
$branches = [];
while ($row = mysqli_fetch_assoc($result)) {
    $branches[] = $row;
}

// Handle filters
$branch_id = isset($_GET['branch_id']) ? $_GET['branch_id'] : null;
$date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d');

// Get daily summary
$dailySummaryResult = $record->getDailySummary($branch_id, $date);
$dailySummary = $dailySummaryResult ? $dailySummaryResult : [];
$totalExpensesResult = $expense->getTotalExpenses($branch_id, $date);
$totalExpenses = $totalExpensesResult ? $totalExpensesResult : 0;
$netProfit = ($dailySummary['total_revenue'] ?? 0) - $totalExpenses;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Daily Reports - Fomax Logistics</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
            margin: 0 auto;
            height: 70px;
        }
        
        .logo {
            display: flex;
            align-items: center;
        }
        
        .logo h1 {
            color: #2a5298;
            font-size: 22px;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #2a5298;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
        }
        
        .nav-links {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
        }
        
        .nav-content {
            display: flex;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .nav-content a {
            padding: 15px 20px;
            text-decoration: none;
            color: #555;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .nav-content a:hover, .nav-content a.active {
            color: #2a5298;
            border-bottom: 3px solid #2a5298;
        }
        
        .container {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .btn {
            background: #2a5298;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            background: #1a2a6c;
        }
        
        .btn.outline {
            background: transparent;
            border: 1px solid #2a5298;
            color: #2a5298;
        }
        
        .btn.outline:hover {
            background: rgba(42, 82, 152, 0.1);
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #555;
        }
        
        input, select {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input:focus, select:focus {
            border-color: #2a5298;
            outline: none;
            box-shadow: 0 0 0 2px rgba(42, 82, 152, 0.2);
        }
        
        .filters {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .filter-actions {
            display: flex;
            gap: 10px;
            margin-top: 10px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            display: flex;
            flex-direction: column;
        }
        
        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
        }
        
        .stat-icon.blue {
            background: rgba(42, 82, 152, 0.1);
            color: #2a5298;
        }
        
        .stat-icon.green {
            background: rgba(40, 167, 69, 0.1);
            color: #28a745;
        }
        
        .stat-icon.red {
            background: rgba(220, 53, 69, 0.1);
            color: #dc3545;
        }
        
        .stat-icon.orange {
            background: rgba(255, 193, 7, 0.1);
            color: #ffc107;
        }
        
        .stat-title {
            font-size: 16px;
            color: #777;
            font-weight: 500;
        }
        
        .stat-value {
            font-size: 28px;
            font-weight: 700;
            margin-top: 5px;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                height: auto;
                padding: 15px 0;
            }
            
            .nav-content {
                flex-wrap: wrap;
            }
            
            .nav-content a {
                padding: 12px 15px;
                font-size: 14px;
            }
            
            .page-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <img src="../images/logo.png" alt="Fomax Logistics" style="height: 40px; margin-right: 15px;">
                <h1>Fomax Logistics</h1>
            </div>
            <div class="user-info">
                <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['username'], 0, 1)); ?></div>
                <div>
                    <div style="font-weight: 500;"><?php echo htmlspecialchars($_SESSION['username']); ?></div>
                    <div style="font-size: 14px; color: #777;">Administrator</div>
                </div>
            </div>
        </div>
    </header>
    
    <nav class="nav-links">
        <div class="nav-content">
            <a href="dashboard.php">
                <span class="material-icons">dashboard</span>
                Dashboard
            </a>
            <a href="branches.php">
                <span class="material-icons">location_city</span>
                Branches
            </a>
            <a href="riders.php">
                <span class="material-icons">local_shipping</span>
                Riders
            </a>
            <a href="users.php">
                <span class="material-icons">people</span>
                Users
            </a>
            <a href="records.php">
                <span class="material-icons">receipt</span>
                Records
            </a>
            <a href="expenses.php">
                <span class="material-icons">account_balance_wallet</span>
                Expenses
            </a>
            <a href="reports.php" class="active">
                <span class="material-icons">assessment</span>
                Reports
            </a>
            <a href="../logout.php">
                <span class="material-icons">exit_to_app</span>
                Logout
            </a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Daily Reports</h1>
        </div>
        
        <div class="card">
            <h3 style="margin-bottom: 20px; color: #2a5298;">Generate Report</h3>
            <form method="GET">
                <div class="filters">
                    <div class="form-group">
                        <label for="branchFilter">Branch</label>
                        <select id="branchFilter" name="branch_id">
                            <option value="">All Branches</option>
                            <?php foreach ($branches as $b): ?>
                                <option value="<?php echo $b['id']; ?>" <?php echo (isset($_GET['branch_id']) && $_GET['branch_id'] == $b['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($b['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="reportDate">Date</label>
                        <input type="date" id="reportDate" name="date" value="<?php echo htmlspecialchars($date); ?>">
                    </div>
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn">Generate Report</button>
                    <a href="reports.php" class="btn outline">Clear Filters</a>
                </div>
            </form>
        </div>
        
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Total Transactions</div>
                        <div class="stat-value"><?php echo $dailySummary['total_transactions'] ?? 0; ?></div>
                    </div>
                    <div class="stat-icon blue">
                        <span class="material-icons">receipt</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Total Revenue</div>
                        <div class="stat-value">₦<?php echo number_format($dailySummary['total_revenue'] ?? 0, 2); ?></div>
                    </div>
                    <div class="stat-icon green">
                        <span class="material-icons">trending_up</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Total Expenses</div>
                        <div class="stat-value">₦<?php echo number_format($totalExpenses, 2); ?></div>
                    </div>
                    <div class="stat-icon red">
                        <span class="material-icons">account_balance_wallet</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Net Profit</div>
                        <div class="stat-value">₦<?php echo number_format($netProfit, 2); ?></div>
                    </div>
                    <div class="stat-icon orange">
                        <span class="material-icons">paid</span>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="card">
            <h3 style="margin-bottom: 20px; color: #2a5298;">
                Report Details 
                <?php if ($branch_id): ?>
                    for <?php 
                        foreach ($branches as $b) {
                            if ($b['id'] == $branch_id) {
                                echo htmlspecialchars($b['name']);
                                break;
                            }
                        }
                    ?>
                <?php else: ?>
                    for All Branches
                <?php endif; ?>
                on <?php echo date('F j, Y', strtotime($date)); ?>
            </h3>
            
            <div style="padding: 20px; background: #f8f9fa; border-radius: 8px;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 15px;">
                    <div>Total Package Value:</div>
                    <div>₦<?php echo number_format($dailySummary['total_price'] ?? 0, 2); ?></div>
                </div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 15px;">
                    <div>Total Delivery Charges:</div>
                    <div>₦<?php echo number_format($dailySummary['total_charges'] ?? 0, 2); ?></div>
                </div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 15px; padding-top: 15px; border-top: 1px solid #ddd;">
                    <div><strong>Total Delivery Charges:</strong></div>
                    <div><strong>₦<?php echo number_format($dailySummary['total_revenue'] ?? 0, 2); ?></strong></div>
                </div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 15px;">
                    <div>Total Expenses:</div>
                    <div>₦<?php echo number_format($totalExpenses, 2); ?></div>
                </div>
                <div style="display: flex; justify-content: space-between; padding-top: 15px; border-top: 2px solid #2a5298;">
                    <div><strong>Net Profit:</strong></div>
                    <div><strong>₦<?php echo number_format($netProfit, 2); ?></strong></div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>