<?php
// admin/riders.php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../models/Rider.php';
require_once '../models/Branch.php';

// Check if user is logged in and is admin
redirectIfNotLoggedIn();
redirectIfNotAdmin();

$database = new Database();
$db = $database->getConnection();
$rider = new Rider($db);
$branch = new Branch($db);

$message = '';
$error = '';

// Create uploads directory if it doesn't exist
$uploadDir = '../uploads/';
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0777, true);
}

// Get all branches for dropdown
$result = $branch->getAll();
$branches = [];
while ($row = mysqli_fetch_assoc($result)) {
    $branches[] = $row;
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $rider->name = trim($_POST['name']);
                $rider->phone_number = trim($_POST['phone_number']);
                $rider->branch_id = $_POST['branch_id'];
                
                // Handle profile picture upload
                $profilePicture = '';
                if (isset($_FILES['profile_picture']) && $_FILES['profile_picture']['error'] == 0) {
                    $fileName = $_FILES['profile_picture']['name'];
                    $fileTmpName = $_FILES['profile_picture']['tmp_name'];
                    $fileSize = $_FILES['profile_picture']['size'];
                    $fileType = $_FILES['profile_picture']['type'];
                    
                    // Validate file type
                    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
                    if (in_array($fileType, $allowedTypes) && $fileSize <= 5000000) { // 5MB limit
                        $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
                        $newFileName = uniqid() . '.' . $fileExtension;
                        $uploadPath = $uploadDir . $newFileName;
                        
                        if (move_uploaded_file($fileTmpName, $uploadPath)) {
                            $profilePicture = 'uploads/' . $newFileName;
                            $rider->profile_picture = $profilePicture;
                        } else {
                            $error = 'Failed to upload profile picture.';
                        }
                    } else {
                        $error = 'Invalid file type or size. Please upload a valid image (JPEG, PNG, GIF) under 5MB.';
                    }
                }
                
                if (empty($rider->name)) {
                    $error = 'Rider name is required.';
                } else if (empty($rider->branch_id)) {
                    $error = 'Please select a branch.';
                } else {
                    if ($rider->create()) {
                        $message = 'Rider created successfully.';
                    } else {
                        $error = 'Failed to create rider.';
                    }
                }
                break;
                
            case 'update':
                $rider->id = $_POST['id'];
                $rider->name = trim($_POST['name']);
                $rider->phone_number = trim($_POST['phone_number']);
                $rider->branch_id = $_POST['branch_id'];
                
                // Handle profile picture upload
                if (isset($_FILES['profile_picture']) && $_FILES['profile_picture']['error'] == 0) {
                    $fileName = $_FILES['profile_picture']['name'];
                    $fileTmpName = $_FILES['profile_picture']['tmp_name'];
                    $fileSize = $_FILES['profile_picture']['size'];
                    $fileType = $_FILES['profile_picture']['type'];
                    
                    // Validate file type
                    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
                    if (in_array($fileType, $allowedTypes) && $fileSize <= 5000000) { // 5MB limit
                        $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
                        $newFileName = uniqid() . '.' . $fileExtension;
                        $uploadPath = $uploadDir . $newFileName;
                        
                        if (move_uploaded_file($fileTmpName, $uploadPath)) {
                            $profilePicture = 'uploads/' . $newFileName;
                            $rider->profile_picture = $profilePicture;
                        } else {
                            $error = 'Failed to upload profile picture.';
                        }
                    } else {
                        $error = 'Invalid file type or size. Please upload a valid image (JPEG, PNG, GIF) under 5MB.';
                    }
                }
                
                if (empty($rider->name)) {
                    $error = 'Rider name is required.';
                } else if (empty($rider->branch_id)) {
                    $error = 'Please select a branch.';
                } else {
                    if ($rider->update()) {
                        $message = 'Rider updated successfully.';
                    } else {
                        $error = 'Failed to update rider.';
                    }
                }
                break;
                
            case 'delete':
                $rider->id = $_POST['id'];
                if ($rider->delete()) {
                    $message = 'Rider deleted successfully.';
                } else {
                    $error = 'Failed to delete rider.';
                }
                break;
        }
    }
}

// Get all riders with branch info
$result = $rider->getAll();
$riders = [];
while ($row = mysqli_fetch_assoc($result)) {
    $riders[] = $row;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Riders - Fomax Logistics</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
            margin: 0 auto;
            height: 70px;
        }
        
        .logo {
            display: flex;
            align-items: center;
        }
        
        .logo h1 {
            color: #2a5298;
            font-size: 22px;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #2a5298;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
        }
        
        .nav-links {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
        }
        
        .nav-content {
            display: flex;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .nav-content a {
            padding: 15px 20px;
            text-decoration: none;
            color: #555;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .nav-content a:hover, .nav-content a.active {
            color: #2a5298;
            border-bottom: 3px solid #2a5298;
        }
        
        .container {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .btn {
            background: #2a5298;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            background: #1a2a6c;
        }
        
        .btn.outline {
            background: transparent;
            border: 1px solid #2a5298;
            color: #2a5298;
        }
        
        .btn.outline:hover {
            background: rgba(42, 82, 152, 0.1);
        }
        
        .btn.danger {
            background: #dc3545;
        }
        
        .btn.danger:hover {
            background: #c82333;
        }
        
        .message {
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #555;
        }
        
        input, select, textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input:focus, select:focus, textarea:focus {
            border-color: #2a5298;
            outline: none;
            box-shadow: 0 0 0 2px rgba(42, 82, 152, 0.2);
        }
        
        .form-actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        .table-container {
            overflow-x: auto;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        th {
            background: #f8f9fa;
            font-weight: 500;
            color: #555;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .actions {
            display: flex;
            gap: 10px;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background: white;
            border-radius: 10px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
        }
        
        .modal-header {
            padding: 20px 25px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-title {
            font-size: 18px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #777;
        }
        
        .modal-body {
            padding: 25px;
        }
        
        .modal-footer {
            padding: 20px 25px;
            border-top: 1px solid #eee;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                height: auto;
                padding: 15px 0;
            }
            
            .nav-content {
                flex-wrap: wrap;
            }
            
            .nav-content a {
                padding: 12px 15px;
                font-size: 14px;
            }
            
            .page-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <img src="../images/logo.png" alt="Fomax Logistics" style="height: 40px; margin-right: 15px;">
                <h1>Fomax Logistics</h1>
            </div>
            <div class="user-info">
                <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['username'], 0, 1)); ?></div>
                <div>
                    <div style="font-weight: 500;"><?php echo htmlspecialchars($_SESSION['username']); ?></div>
                    <div style="font-size: 14px; color: #777;">Administrator</div>
                </div>
            </div>
        </div>
    </header>
    
    <nav class="nav-links">
        <div class="nav-content">
            <a href="dashboard.php">
                <span class="material-icons">dashboard</span>
                Dashboard
            </a>
            <a href="branches.php">
                <span class="material-icons">location_city</span>
                Branches
            </a>
            <a href="riders.php" class="active">
                <span class="material-icons">local_shipping</span>
                Riders
            </a>
            <a href="users.php">
                <span class="material-icons">people</span>
                Users
            </a>
            <a href="records.php">
                <span class="material-icons">receipt</span>
                Records
            </a>
            <a href="expenses.php">
                <span class="material-icons">account_balance_wallet</span>
                Expenses
            </a>
            <a href="reports.php">
                <span class="material-icons">assessment</span>
                Reports
            </a>
            <a href="../logout.php">
                <span class="material-icons">exit_to_app</span>
                Logout
            </a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Manage Riders</h1>
            <button class="btn" id="addRiderBtn">
                <span class="material-icons" style="font-size: 18px; margin-right: 5px;">add</span>
                Add New Rider
            </button>
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="card">
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Profile</th>
                            <th>Name</th>
                            <th>Phone Number</th>
                            <th>Branch</th>
                            <th>Created At</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($riders)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center;">No riders found</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($riders as $r): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($r['id']); ?></td>
                                    <td>
                                        <?php if (!empty($r['profile_picture'])): ?>
                                            <img src="../<?php echo htmlspecialchars($r['profile_picture']); ?>" alt="Profile" style="width: 40px; height: 40px; border-radius: 50%; object-fit: cover;">
                                        <?php else: ?>
                                            <div style="width: 40px; height: 40px; border-radius: 50%; background: #eee; display: flex; align-items: center; justify-content: center;">
                                                <span class="material-icons">person</span>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($r['name']); ?></td>
                                    <td><?php echo htmlspecialchars($r['phone_number']); ?></td>
                                    <td><?php echo htmlspecialchars($r['branch_name']); ?></td>
                                    <td><?php echo date('M j, Y', strtotime($r['created_at'])); ?></td>
                                    <td class="actions">
                                        <button class="btn outline edit-btn" 
                                            data-id="<?php echo $r['id']; ?>" 
                                            data-name="<?php echo htmlspecialchars($r['name']); ?>" 
                                            data-phone="<?php echo htmlspecialchars($r['phone_number']); ?>" 
                                            data-branch="<?php echo $r['branch_id']; ?>"
                                            data-profile="<?php echo htmlspecialchars($r['profile_picture']); ?>">
                                            <span class="material-icons" style="font-size: 16px;">edit</span>
                                        </button>
                                        <button class="btn danger delete-btn" 
                                            data-id="<?php echo $r['id']; ?>" 
                                            data-name="<?php echo htmlspecialchars($r['name']); ?>">
                                            <span class="material-icons" style="font-size: 16px;">delete</span>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Rider Modal -->
    <div class="modal" id="riderModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title" id="modalTitle">Add New Rider</div>
                <button class="close-modal">&times;</button>
            </div>
            <form method="POST" id="riderForm" enctype="multipart/form-data">
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="id" id="riderId">
                <div class="modal-body">
                    <div class="form-group">
                        <label for="riderName">Rider Name</label>
                        <input type="text" id="riderName" name="name" required>
                    </div>
                    <div class="form-group">
                        <label for="riderPhone">Phone Number</label>
                        <input type="text" id="riderPhone" name="phone_number">
                    </div>
                    <div class="form-group">
                        <label for="riderBranch">Branch</label>
                        <select id="riderBranch" name="branch_id" required>
                            <option value="">Select a branch</option>
                            <?php foreach ($branches as $b): ?>
                                <option value="<?php echo $b['id']; ?>"><?php echo htmlspecialchars($b['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="profilePicture">Profile Picture</label>
                        <input type="file" id="profilePicture" name="profile_picture" accept="image/*">
                        <div id="currentPicture" style="margin-top: 10px;"></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn outline closeModal">Cancel</button>
                    <button type="submit" class="btn">Save Rider</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Delete Confirmation Modal -->
    <div class="modal" id="deleteModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">Confirm Delete</div>
                <button class="close-modal">&times;</button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the rider "<span id="deleteRiderName"></span>"? This action cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn outline closeModal">Cancel</button>
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="deleteRiderId">
                    <button type="submit" class="btn danger">Delete Rider</button>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        // Modal elements
        const riderModal = document.getElementById('riderModal');
        const deleteModal = document.getElementById('deleteModal');
        const addRiderBtn = document.getElementById('addRiderBtn');
        const closeModalButtons = document.querySelectorAll('.closeModal, .close-modal');
        const profilePictureInput = document.getElementById('profilePicture');
        const currentPicture = document.getElementById('currentPicture');
        
        // Form elements
        const formAction = document.getElementById('formAction');
        const riderId = document.getElementById('riderId');
        const riderName = document.getElementById('riderName');
        const riderPhone = document.getElementById('riderPhone');
        const riderBranch = document.getElementById('riderBranch');
        const modalTitle = document.getElementById('modalTitle');
        
        // Delete elements
        const deleteRiderId = document.getElementById('deleteRiderId');
        const deleteRiderName = document.getElementById('deleteRiderName');
        
        // Event listeners
        addRiderBtn.addEventListener('click', () => {
            // Reset form
            formAction.value = 'create';
            riderId.value = '';
            riderName.value = '';
            riderPhone.value = '';
            riderBranch.value = '';
            profilePictureInput.value = '';
            currentPicture.innerHTML = '';
            modalTitle.textContent = 'Add New Rider';
            riderModal.style.display = 'flex';
        });
        
        // Edit buttons
        document.querySelectorAll('.edit-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const id = e.target.closest('.edit-btn').dataset.id;
                const name = e.target.closest('.edit-btn').dataset.name;
                const phone = e.target.closest('.edit-btn').dataset.phone;
                const branch = e.target.closest('.edit-btn').dataset.branch;
                const profile = e.target.closest('.edit-btn').dataset.profile;
                
                // Fill form
                formAction.value = 'update';
                riderId.value = id;
                riderName.value = name;
                riderPhone.value = phone;
                riderBranch.value = branch;
                
                // Handle profile picture
                profilePictureInput.value = '';
                if (profile) {
                    currentPicture.innerHTML = `<img src="../${profile}" alt="Current Profile" style="width: 80px; height: 80px; border-radius: 50%; object-fit: cover;">`;
                } else {
                    currentPicture.innerHTML = '';
                }
                
                modalTitle.textContent = 'Edit Rider';
                riderModal.style.display = 'flex';
            });
        });
        
        // Delete buttons
        document.querySelectorAll('.delete-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const id = e.target.closest('.delete-btn').dataset.id;
                const name = e.target.closest('.delete-btn').dataset.name;
                
                // Set delete form values
                deleteRiderId.value = id;
                deleteRiderName.textContent = name;
                deleteModal.style.display = 'flex';
            });
        });
        
        // Close modals
        closeModalButtons.forEach(button => {
            button.addEventListener('click', () => {
                riderModal.style.display = 'none';
                deleteModal.style.display = 'none';
            });
        });
        
        // Close modal when clicking outside
        window.addEventListener('click', (e) => {
            if (e.target === riderModal) {
                riderModal.style.display = 'none';
            }
            if (e.target === deleteModal) {
                deleteModal.style.display = 'none';
            }
        });
    </script>
</body>
</html>