<?php
// admin/dashboard.php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../models/Branch.php';
require_once '../models/Record.php';
require_once '../models/Expense.php';

// Check if user is logged in and is admin
redirectIfNotLoggedIn();
if (!isAdmin()) {
    header('Location: ../user/dashboard.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Get today's date
$today = date('Y-m-d');

// Get dashboard statistics
$record = new Record($db);
$expense = new Expense($db);

// Today's summary
$todaySummary = $record->getDailySummary(null, $today);
$totalExpenses = $expense->getTotalExpenses(null, $today);

// Payment mode summary
$paymentSummary = $record->getPaymentModeSummary(null, $today);
$totalCash = isset($paymentSummary['Cash']) ? $paymentSummary['Cash']['amount'] : 0;
$totalTransfers = 0;
foreach ($paymentSummary as $mode => $data) {
    if ($mode != 'Cash') {
        $totalTransfers += $data['amount'];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Fomax Logistics</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
            margin: 0 auto;
            height: 70px;
        }
        
        .logo {
            display: flex;
            align-items: center;
        }
        
        .logo h1 {
            color: #2a5298;
            font-size: 22px;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #2a5298;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
        }
        
        .nav-links {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
        }
        
        .nav-content {
            display: flex;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .nav-content a {
            padding: 15px 20px;
            text-decoration: none;
            color: #555;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .nav-content a:hover, .nav-content a.active {
            color: #2a5298;
            border-bottom: 3px solid #2a5298;
        }
        
        .container {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 500;
            margin-bottom: 25px;
            color: #2a5298;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            display: flex;
            flex-direction: column;
        }
        
        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
        }
        
        .stat-icon.blue {
            background: rgba(42, 82, 152, 0.1);
            color: #2a5298;
        }
        
        .stat-icon.green {
            background: rgba(40, 167, 69, 0.1);
            color: #28a745;
        }
        
        .stat-icon.red {
            background: rgba(220, 53, 69, 0.1);
            color: #dc3545;
        }
        
        .stat-icon.orange {
            background: rgba(255, 193, 7, 0.1);
            color: #ffc107;
        }
        
        .stat-title {
            font-size: 16px;
            color: #777;
            font-weight: 500;
        }
        
        .stat-value {
            font-size: 28px;
            font-weight: 700;
            margin-top: 5px;
        }
        
        .content-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .card-title {
            font-size: 18px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .btn {
            background: #2a5298;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            background: #1a2a6c;
        }
        
        .btn.outline {
            background: transparent;
            border: 1px solid #2a5298;
            color: #2a5298;
        }
        
        .btn.outline:hover {
            background: rgba(42, 82, 152, 0.1);
        }
        
        .quick-links {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .link-card {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            transition: all 0.3s;
            text-decoration: none;
            color: inherit;
            display: block;
        }
        
        .link-card:hover {
            background: #e9ecef;
            transform: translateY(-3px);
        }
        
        .link-icon {
            font-size: 32px;
            color: #2a5298;
            margin-bottom: 15px;
        }
        
        .link-title {
            font-weight: 500;
            color: #333;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                height: auto;
                padding: 15px 0;
            }
            
            .nav-content {
                flex-wrap: wrap;
            }
            
            .nav-content a {
                padding: 12px 15px;
                font-size: 14px;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <img src="../images/logo.png" alt="Fomax Logistics" style="height: 40px; margin-right: 15px;">
                <h1>Fomax Logistics</h1>
            </div>
            <div class="user-info">
                <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['username'], 0, 1)); ?></div>
                <div>
                    <div style="font-weight: 500;"><?php echo htmlspecialchars($_SESSION['username']); ?></div>
                    <div style="font-size: 14px; color: #777;">Administrator</div>
                </div>
            </div>
        </div>
    </header>
    
    <nav class="nav-links">
        <div class="nav-content">
            <a href="dashboard.php" class="active">
                <span class="material-icons">dashboard</span>
                Dashboard
            </a>
            <a href="branches.php">
                <span class="material-icons">location_city</span>
                Branches
            </a>
            <a href="riders.php">
                <span class="material-icons">local_shipping</span>
                Riders
            </a>
            <a href="users.php">
                <span class="material-icons">people</span>
                Users
            </a>
            <a href="records.php">
                <span class="material-icons">receipt</span>
                Records
            </a>
            <a href="expenses.php">
                <span class="material-icons">account_balance_wallet</span>
                Expenses
            </a>
            <a href="reports.php">
                <span class="material-icons">assessment</span>
                Reports
            </a>
            <a href="../logout.php">
                <span class="material-icons">exit_to_app</span>
                Logout
            </a>
        </div>
    </nav>
    
    <div class="container">
        <h1 class="page-title">Admin Dashboard</h1>
        
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Today's Transactions</div>
                        <div class="stat-value"><?php echo $todaySummary['total_transactions'] ?? 0; ?></div>
                    </div>
                    <div class="stat-icon blue">
                        <span class="material-icons">receipt</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Cash Collected</div>
                        <div class="stat-value">₦<?php echo number_format($totalCash, 2); ?></div>
                    </div>
                    <div class="stat-icon green">
                        <span class="material-icons">attach_money</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Bank Transfers</div>
                        <div class="stat-value">₦<?php echo number_format($totalTransfers, 2); ?></div>
                    </div>
                    <div class="stat-icon blue">
                        <span class="material-icons">account_balance</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Today's Delivery Charges</div>
                        <div class="stat-value">₦<?php echo number_format($todaySummary['total_revenue'] ?? 0, 2); ?></div>
                    </div>
                    <div class="stat-icon green">
                        <span class="material-icons">trending_up</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Today's Expenses</div>
                        <div class="stat-value">₦<?php echo number_format($totalExpenses, 2); ?></div>
                    </div>
                    <div class="stat-icon red">
                        <span class="material-icons">account_balance_wallet</span>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-title">Net Profit</div>
                        <div class="stat-value">₦<?php 
                            $netProfit = ($todaySummary['total_revenue'] ?? 0) - $totalExpenses;
                            echo number_format($netProfit, 2);
                        ?></div>
                    </div>
                    <div class="stat-icon orange">
                        <span class="material-icons">paid</span>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="content-grid">
            <div class="card">
                <div class="card-header">
                    <div class="card-title">Quick Actions</div>
                </div>
                <div class="quick-links">
                    <a href="branches.php" class="link-card">
                        <div class="link-icon">
                            <span class="material-icons">location_city</span>
                        </div>
                        <div class="link-title">Manage Branches</div>
                    </a>
                    
                    <a href="riders.php" class="link-card">
                        <div class="link-icon">
                            <span class="material-icons">local_shipping</span>
                        </div>
                        <div class="link-title">Manage Riders</div>
                    </a>
                    
                    <a href="users.php" class="link-card">
                        <div class="link-icon">
                            <span class="material-icons">people</span>
                        </div>
                        <div class="link-title">Manage Users</div>
                    </a>
                    
                    <a href="reports.php" class="link-card">
                        <div class="link-icon">
                            <span class="material-icons">assessment</span>
                        </div>
                        <div class="link-title">View Reports</div>
                    </a>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <div class="card-title">Recent Activity</div>
                </div>
                <div style="padding: 20px; text-align: center; color: #777;">
                    <span class="material-icons" style="font-size: 48px; margin-bottom: 15px;">history</span>
                    <p>No recent activity to display</p>
                </div>
            </div>
        </div>
    </div>
</body>
</html>