<?php
// user/add_record.php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../models/Record.php';
require_once '../models/Rider.php';

// Check if user is logged in
redirectIfNotLoggedIn();
if (isAdmin()) {
    header('Location: ../admin/dashboard.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

$message = '';
$error = '';

// Get riders for this user's branch
$rider = new Rider($db);
$result = $rider->getByBranchId($_SESSION['branch_id']);
$riders = [];
while ($row = mysqli_fetch_assoc($result)) {
    $riders[] = $row;
}

$paymentModes = ['Cash', 'POS'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $record = new Record($db);
    
    $record->rider_id = $_POST['rider_id'];
    $record->package_description = trim($_POST['package_description']);
    $record->quantity = $_POST['quantity'];
    $record->item_valuation = $_POST['item_valuation'];
    $record->location = trim($_POST['location']);
    $record->vendor_charges = $_POST['vendor_charges'];
    $record->payment_mode = $_POST['payment_mode'];
    $record->notes = trim($_POST['notes']);
    $record->user_id = $_SESSION['user_id'];
    $record->branch_id = $_SESSION['branch_id'];
    
    if (empty($record->rider_id)) {
        $error = 'Please select a rider.';
    } else if (empty($record->package_description)) {
        $error = 'Package description is required.';
    } else if (empty($record->quantity) || $record->quantity <= 0) {
        $error = 'Quantity must be a positive number.';
    } else if (empty($record->item_valuation) || $record->item_valuation < 0) {
        $error = 'Item valuation must be a valid number.';
    } else if (empty($record->location)) {
        $error = 'Delivery location is required.';
    } else if (empty($record->payment_mode)) {
        $error = 'Payment mode is required.';
    } else {
        $recordId = $record->create();
        if ($recordId) {
            $message = 'Record added successfully.';
            // Reset form values
            $record->rider_id = '';
            $record->package_description = '';
            $record->quantity = '';
            $record->item_valuation = '';
            $record->location = '';
            $record->vendor_charges = '';
            $record->payment_mode = '';
            $record->notes = '';
        } else {
            $error = 'Failed to add record.';
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Record - Fomax Logistics</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1200px;
            margin: 0 auto;
            height: 70px;
        }
        
        .logo {
            display: flex;
            align-items: center;
        }
        
        .logo h1 {
            color: #2a5298;
            font-size: 22px;
            font-weight: 700;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #2a5298;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 500;
        }
        
        .nav-links {
            background: white;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 0 20px;
        }
        
        .nav-content {
            display: flex;
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .nav-content a {
            padding: 15px 20px;
            text-decoration: none;
            color: #555;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: color 0.3s;
            border-bottom: 3px solid transparent;
        }
        
        .nav-content a:hover, .nav-content a.active {
            color: #2a5298;
            border-bottom: 3px solid #2a5298;
        }
        
        .container {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .page-title {
            font-size: 24px;
            font-weight: 500;
            color: #2a5298;
        }
        
        .btn {
            background: #2a5298;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background 0.3s;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            background: #1a2a6c;
        }
        
        .btn.outline {
            background: transparent;
            border: 1px solid #2a5298;
            color: #2a5298;
        }
        
        .btn.outline:hover {
            background: rgba(42, 82, 152, 0.1);
        }
        
        .message {
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #555;
        }
        
        input, select, textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input:focus, select:focus, textarea:focus {
            border-color: #2a5298;
            outline: none;
            box-shadow: 0 0 0 2px rgba(42, 82, 152, 0.2);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .form-actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                height: auto;
                padding: 15px 0;
            }
            
            .nav-content {
                flex-wrap: wrap;
            }
            
            .nav-content a {
                padding: 12px 15px;
                font-size: 14px;
            }
            
            .page-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
        }
    </style>
</head>
<body>
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <img src="../images/logo.png" alt="Fomax Logistics" style="height: 40px; margin-right: 15px;">
                <h1>Fomax Logistics</h1>
            </div>
            <div class="user-info">
                <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['username'], 0, 1)); ?></div>
                <div>
                    <div style="font-weight: 500;"><?php echo htmlspecialchars($_SESSION['username']); ?></div>
                    <div style="font-size: 14px; color: #777;">Staff User</div>
                </div>
            </div>
        </div>
    </header>
    
    <nav class="nav-links">
        <div class="nav-content">
            <a href="dashboard.php">
                <span class="material-icons">dashboard</span>
                Dashboard
            </a>
            <a href="add_record.php" class="active">
                <span class="material-icons">add_circle</span>
                Add Record
            </a>
            <a href="add_expense.php">
                <span class="material-icons">add_shopping_cart</span>
                Add Expense
            </a>
            <a href="my_records.php">
                <span class="material-icons">list</span>
                My Records
            </a>
            <a href="../logout.php">
                <span class="material-icons">exit_to_app</span>
                Logout
            </a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Add New Record</h1>
        </div>
        
        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="card">
            <form method="POST">
                <div class="form-group">
                    <label for="riderId">Rider's Name</label>
                    <select id="riderId" name="rider_id" required>
                        <option value="">Select a rider</option>
                        <?php foreach ($riders as $r): ?>
                            <option value="<?php echo $r['id']; ?>" <?php echo (isset($_POST['rider_id']) && $_POST['rider_id'] == $r['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($r['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="packageName">Package Description</label>
                    <input type="text" id="packageName" name="package_description" 
                           value="<?php echo isset($_POST['product_name']) ? htmlspecialchars($_POST['product_name']) : ''; ?>" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="quantity">Quantity</label>
                        <input type="number" id="quantity" name="quantity" min="1" 
                               value="<?php echo isset($_POST['quantity']) ? htmlspecialchars($_POST['quantity']) : ''; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="price">Product Price (₦)</label>
                        <input type="number" id="price" name="item_valuation" step="0.01" min="0" 
                               value="<?php echo isset($_POST['item_valuation']) ? htmlspecialchars($_POST['item_valuation']) : ''; ?>" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="location">Delivery Location</label>
                    <textarea id="location" name="location" rows="2" required><?php echo isset($_POST['location']) ? htmlspecialchars($_POST['location']) : ''; ?></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="vendorCharges">Delivery Charges (₦)</label>
                        <input type="number" id="vendorCharges" name="vendor_charges" step="0.01" min="0" 
                               value="<?php echo isset($_POST['vendor_charges']) ? htmlspecialchars($_POST['vendor_charges']) : '0'; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="paymentMode">Payment Mode</label>
                        <select id="paymentMode" name="payment_mode" required>
                            <option value="">Select payment mode</option>
                            <?php foreach ($paymentModes as $mode): ?>
                                <option value="<?php echo $mode; ?>" <?php echo (isset($_POST['payment_mode']) && $_POST['payment_mode'] == $mode) ? 'selected' : ''; ?>>
                                    <?php echo $mode; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="notes">Notes (Optional)</label>
                    <textarea id="notes" name="notes" rows="3"><?php echo isset($_POST['notes']) ? htmlspecialchars($_POST['notes']) : ''; ?></textarea>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn">Add Record</button>
                    <a href="dashboard.php" class="btn outline">Cancel</a>
                </div>
            </form>
        </div>
    </div>
</body>
</html>